<?php

/**
 * This is the model class for table "SzkolenieFirmy".
 *
 * The followings are the available columns in table 'SzkolenieFirmy':
 * @property double $id
 * @property double $idFirmy
 * @property double $idSzkolenia
 * @property string $czySukces
 * @property string $dataRozpoczecia
 * @property string $dataZakonczenia
 * @property string $idFaktury
 * @property string $zgloszone
 * @property string $idUcznia
 *
 * @property Firma $idFirmy0
 * @property Szkolenie $idSzkolenia0
 * @property SzkolenieOdpowiedziFirmy[] $pytaniaEgzaminacyjne
 * @property SzkolenieFirmyOdp[] $zleceniaEgzaminacyjne
 */
class SzkolenieFirmy extends CActiveRecord
{

	/**
	 * Returns the static model of the specified AR class.
	 * @param string $className active record class name.
	 * @return SzkolenieFirmy the static model class
	 */
	public static function model($className = __CLASS__)
	{
		return parent::model($className);
	}

	/**
	 * @return string the associated database table name
	 */
	public function tableName()
	{
		return 'SzkolenieFirmy';
	}

	/**
	 * @return array validation rules for model attributes.
	 */
	public function rules()
	{
		return array(
			array('idFirmy, idSzkolenia', 'required'),
			array('idFirmy, idSzkolenia', 'numerical'),
			array('czySukces', 'length', 'max' => 1),
			array('dataRozpoczecia, dataZakonczenia, zgloszone, idFaktury, czySukces, idUcznia', 'safe'),
			array('id, idFirmy, idSzkolenia, czySukces, dataRozpoczecia, dataZakonczenia',
				'safe', 'on' => 'search'),
		);
	}

	/**
	 * @return array relational rules.
	 */
	public function relations()
	{
		return array(
			'idFirmy0' => array(self::BELONGS_TO, 'Firma', 'idFirmy'),
			'idSzkolenia0' => array(self::BELONGS_TO, 'Szkolenie', 'idSzkolenia'),
			'pytaniaEgzaminacyjne' => array(self::HAS_MANY, 'SzkolenieOdpowiedziFirmy', 'idSzkolenieFirmy'),
			'zleceniaEgzaminacyjne' => array(self::HAS_MANY, 'SzkolenieFirmyOdp', 'idSzkolenieFirmy'),
			'faktura' => array(self::BELONGS_TO, 'Faktury', 'idFaktury'),
			'uzytkownik' => array(self::BELONGS_TO, 'Uzytkownik', 'idUcznia'),
		);
	}

	/**
	 * @return array customized attribute labels (name=>label)
	 */
	public function attributeLabels()
	{
		return array(
			'id' => Yii::t('SzkolenieFirmy', 'id'),
			'idFirmy' => Yii::t('SzkolenieFirmy', 'idFirmy'),
			'idSzkolenia' => Yii::t('SzkolenieFirmy', 'idSzkolenia'),
			'czySukces' => Yii::t('SzkolenieFirmy', 'czySukces'),
			'dataRozpoczecia' => Yii::t('SzkolenieFirmy', 'dataRozpoczecia'),
			'dataZakonczenia' => Yii::t('SzkolenieFirmy', 'dataZakonczenia'),
		);
	}

	/**
	 * Retrieves a list of models based on the current search/filter conditions.
	 * @return CActiveDataProvider the data provider that can return the models based on the search/filter conditions.
	 */
	public function search()
	{
		$criteria = new CDbCriteria;

		$criteria->compare('id', $this->id);
		$criteria->compare('idFirmy', $this->idFirmy);
		$criteria->compare('idSzkolenia', $this->idSzkolenia);
		$criteria->compare('czySukces', $this->czySukces, true);
		$criteria->compare('dataRozpoczecia', $this->dataRozpoczecia, true);
		$criteria->compare('dataZakonczenia', $this->dataZakonczenia, true);

		return new CActiveDataProvider($this, array(
			'criteria' => $criteria,
		));
	}

	public function getDataZakonczeniaYmd()
	{
		return empty($this->dataZakonczenia) ? '' : date('Y/m/d',
				strtotime($this->dataZakonczenia));
	}

	/**
	 * Wystawia fakture za szkolenie
	 * @staticvar array $csKontrahent idDziedziny -> idKontrahenta
	 * @param Szkolenie $szkolenie Opcjonalnie obiekt szkolenia
	 * @return int id faktury, null jak brak
	 */
	public function wystawFakture(Szkolenie $szkolenie = null)
	{
		if ($szkolenie == null)
			$szkolenie = $this->idSzkolenia0;
		static $csKontrahent = array(
			1 => 48, 2 => 47, 3 => 49, 4 => 50,
		);
		$faktura = new Faktury;
		$faktura->czyKoszt = 1;
		$faktura->idFirmy = Yii::app()->user->id_firmy;
		$faktura->idKontrahenta = $csKontrahent[$szkolenie->idDziedzina];
		$faktura->dataWystawienia = new CDbExpression('SYSDATE');
		$faktura->opis = 'Szkolenie ' . $szkolenie->dziedzina->nazwa
			. ' - ' . $szkolenie->nazwa . ' (poziom ' . $szkolenie->poziom . ')';
		$faktura->nrFaktury = 'F/' . date('Y/m/d/', time()) . rand(0, 9) . rand(0, 9) . rand(0, 9);
		$faktura->stawkaVat = ParametryGry::getFloat('PODATEK_VAT', 0.23);
		$faktura->cenaNetto = $szkolenie->cena;
		
		$company = Firma::model()->findByPk(Yii::app()->user->id_firmy);
		$company->payOrder($faktura);
		$faktura->save(false);
		$company->save(false);
		return $faktura->id;
	}

	private function accept(Firma &$company, array $points)
	{
		if ($this->czySukces)
		{
			$company->transferPointsWithWorkers($this->idSzkolenia0->idDziedzina, $points[$this->idSzkolenia0->poziom]);

			foreach ($company->activeWorkers as $worker)
			{
				$cert = new SzkoleniePracownika;
				$cert->dataSzkolenia = $this->dataZakonczenia;
				$cert->idPracownika = $worker->id;
				$cert->idSzkolenia = $this->idSzkolenia;
				$cert->save(false);
			}
		}
	}

	/**
	 * Wykupuje szkolenie
	 * @param type $szkolenieId id szkolenie
	 * @param type $begin czy automatycznie rozpoczać egzamin (o ile szkolenie ma egzamin)
	 * @return array
	 */
	public static function wykup(Firma &$company, $szkolenieId, $begin = false)
	{
		$result = array();
		$transaction = Yii::app()->db->beginTransaction();
		try
		{
			//znajdź szkolenie
			$SZKOLENIE = Szkolenie::model()->findByPk($szkolenieId);
			if (empty($SZKOLENIE))
				throw new Exception('Nie ma takiego szkolenia.');

			//znajdź maksymalny poziom
			$maxPoziom = Yii::app()->db->createCommand('
						select "s"."poziom" from "SzkolenieFirmy" "t"
						left join "Szkolenie" "s" ON "t"."idSzkolenia" = "s"."id"
						where "t"."idFirmy" = :idFirmy AND "s"."idDziedzina" = :idDziedziny
						group by "s"."poziom"
						having SUM("t"."czySukces") >= :treshold
						order by "s"."poziom" DESC
			')->queryScalar(array(
				':idFirmy' => $company->id,
				':idDziedziny' => $SZKOLENIE->idDziedzina,
				':treshold' => 1
			));

			//sprawdź czy poziom jest wystarczający
			if ($SZKOLENIE->poziom > $maxPoziom + 1)
				throw new Exception('Twoja firma nie osiągnęła wystarczającego poziomu by podjąć się tego szkolenia.');

			//dodaj szkolenie
			$szkolenie = new SzkolenieFirmy;
			$szkolenie->idFirmy = $company->id;
			$szkolenie->idSzkolenia = $szkolenieId;
			$szkolenie->idUcznia = Yii::app()->user->id;
			$szkolenie->zgloszone = null;
			if ($SZKOLENIE->czyEgzamin)
			{
				$szkolenie->czySukces = null;
				if ($begin)	//automatyczni rozpocznij egzamin
				{
					$beginTs = time();
					$endTs = $beginTs + $SZKOLENIE->czasTrwania;
					$szkolenie->dataRozpoczecia = date('d-m-Y H:i:s', $beginTs);
					$szkolenie->dataZakonczenia = date('d-m-Y H:i:s', $endTs);
				}
				else
				{
					$szkolenie->dataRozpoczecia = null;
					$szkolenie->dataZakonczenia = null;
				}
			}
			else
			{
				//szkolenia bez egazminu są automatycznie zalicznae podczas pierwszego czytania
				$szkolenie->czySukces = 1;
				$szkolenie->dataRozpoczecia = new CDbExpression('SYSDATE');
				$szkolenie->dataZakonczenia = new CDbExpression('SYSDATE');
				$szkolenie->accept($company, array(
					Zlecenie::LEVEL_EASY => ParametryGry::getFloat('PUNKTY_SZKOLENIE_LATWE', 5),
					Zlecenie::LEVEL_HARD => ParametryGry::getFloat('PUNKTY_SZKOLENIE_SREDNIE', 15),
					Zlecenie::LEVEL_VERY_HARD => ParametryGry::getFloat('PUNKTY_SZKOLENIE_TRUDNE', 50)
				));
			}
			if (!$szkolenie->save())
				throw new Exception('Problem podczas wykupowania szkolenia.');

			//wylosuj pytania
			if ($SZKOLENIE->czyEgzamin)
			{
				$PYTANIA = SzkoleniePytaniaEgzamin::model()->findAllBySql('
					SELECT * FROM
					(
						select * from "SzkoleniePytaniaEgzamin" "spe"
						where "spe"."idSzkolenia" = :idSzkolenia
						ORDER BY dbms_random.value
					)
					WHERE rownum <= :max
				', array(
					':max' => (int)ParametryGry::getValue('EGZAMIN_ILOSC_PYTAN', 5),
					':idSzkolenia' => $SZKOLENIE->id
				));
				//zapisz pytania
				foreach ($PYTANIA as $PYTANIE)
				{
					$pytanie = new SzkolenieOdpowiedziFirmy;
					$pytanie->idSzkolenieFirmy = $szkolenie->id;
					$pytanie->idPytania = $PYTANIE->id;
					if (!$pytanie->save())
						throw new Exception('Problem podczas losowania pytań');
					else if ($begin)
						$result['pytania'][] = array(
							'szkolenieId' => $pytanie->idSzkolenieFirmy,
							'pytanieId' => $pytanie->idPytania,
							'wartosc' => '',
							'tresc' => $PYTANIE->tresc,
							'urlObrazka' => $PYTANIE->urlObrazka
						);
				}
			}

			$result['szkolenie'] = $szkolenie->attributes + $SZKOLENIE->attributes;
			$result['szkolenie']['czySukces'] = 0;
			$result['szkolenie']['faktura'] = $szkolenie->wystawFakture($SZKOLENIE);
			$result['szkolenie']['poziomStr'] = $SZKOLENIE->getPoziomStr();
			
			$szkolenie->idFaktury = $result['szkolenie']['faktura'];
			$szkolenie->save(false);

			$transaction->commit();
		}
		catch (Exception $e)
		{
			$transaction->rollback();
			$result['error'] = $e->getMessage();
		}
		return $result;
	}

	public function checkExam(Firma &$company, $answers = null)
	{
		$vaildCount = 0;
		foreach($this->zleceniaEgzaminacyjne as $pytEx)
		{
			$odp = is_array($answers) && isset($answers[$pytEx->id]) ?
					$answers[$pytEx->id] : $pytEx->odpowiedz;
			$pytEx->idOdpowiedzi = $pytEx->zlecenie->checkAnswer($odp);
			if ($pytEx->idOdpowiedzi)
				$vaildCount++;
			$pytEx->save();
		}
		$progZdawalnosci = ParametryGry::getInt('EGZAMIN_PROG_ZDAWALNOSCI', 3);
		$this->czySukces = $vaildCount >= $progZdawalnosci ? 1 : 0;
		$this->accept($company, array(
			Zlecenie::LEVEL_EASY => ParametryGry::getFloat('PUNKTY_EGZAMIN_LATWY', 10),
			Zlecenie::LEVEL_HARD => ParametryGry::getFloat('PUNKTY_EGZAMIN_SREDNI', 50),
			Zlecenie::LEVEL_VERY_HARD => ParametryGry::getFloat('PUNKTY_EGZAMIN_TRUDNY', 100)
		));
		if (!$this->save())
			throw new Exception('Problem podczas zapisywania szkolenia.');
		return $vaildCount;
	}

	public static function checkPendingExams(Firma &$company)
	{
		$now = date('d-m-Y H:i:s');
		
		$exams = SzkolenieFirmy::model()->with(array('pytaniaEgzaminacyjne.pytanie.odpowiedzi'))
			->findAll('"t"."czySukces" is NULL and "t"."dataRozpoczecia" IS NOT NULL AND
				"t"."dataZakonczenia" < :now AND "t"."idFirmy" = :idFirmy',
			array(
				':idFirmy' => $company->id,
				':now' => $now
			));

		foreach ($exams as $exam) {
			$totalCount = 0;
			$vaildCount = 0;
			foreach($exam->pytaniaEgzaminacyjne as $pytEx)
			{
				$odpowiedz = trim(strtolower($pytEx->wartosc));
				foreach ($pytEx->pytanie->odpowiedzi as $odp)
					if ($odpowiedz == trim(strtolower($odp->tresc)))
					{
						$pytEx->idOdpowiedzi = $odp->id;
						if ($pytEx->save())
							$vaildCount++;
					}
				$totalCount++;
			}
			$exam->czySukces = $vaildCount >= (int)ParametryGry::getValue('EGZAMIN_PROG_ZDAWALNOSCI', 3) ? 1 : 0;
			$exam->save();
		}
	}
}
